'use client';

import React, { useState, useEffect } from 'react';
import { useParams } from 'next/navigation';
import { 
  User, 
  Building2, 
  Phone, 
  Mail, 
  MapPin, 
  Calendar, 
  Crown, 
  Edit,
  Settings,
  Plane,
  Hotel,
  Globe,
  Car,
  CreditCard,
  FileText,
  Users,
  Plus,
  TrendingUp,
  Clock
} from 'lucide-react';
import Link from 'next/link';
import DetailLayout from '@/components/ui/DetailLayout';
import { renderStatusBadge, renderCurrency, renderServiceType } from '@/components/ui/TableRenderers';
import type { Customer } from '@/types';
import PermissionGuard from '@/components/guards/PermissionGuard';

interface Comment {
  id: string;
  author: string;
  content: string;
  timestamp: Date;
  type: 'comment' | 'system' | 'status_change';
  isInternal?: boolean;
}

interface CustomerService {
  id: string;
  type: 'ticket' | 'accommodation' | 'visa' | 'transfer';
  description: string;
  amount: number;
  currency: string;
  status: string;
  date: Date;
}

export default function CustomerDetailPage() {
  return (
    <PermissionGuard requiredPermissions={['customers_view', 'customers_view_own']}>
      <CustomerDetailPageContent />
    </PermissionGuard>
  );
}

function CustomerDetailPageContent() {
  const params = useParams();
  const customerId = params.id as string;
  
  const [customer, setCustomer] = useState<Customer | null>(null);
  const [loading, setLoading] = useState(true);
  const [comments, setComments] = useState<Comment[]>([]);
  const [services, setServices] = useState<CustomerService[]>([]);
  const [activeTab, setActiveTab] = useState<'overview' | 'services' | 'users'>('overview');

  useEffect(() => {
    fetchCustomerDetails();
    fetchComments();
    fetchServices();
  }, [customerId]);

  const fetchCustomerDetails = async () => {
    try {
      setLoading(true);
      
      // Mock data
      const mockCustomer: Customer = {
        id: customerId,
        type: 'corporate',
        companyName: 'ABC Teknoloji A.Ş.',
        taxNumber: '1234567890',
        taxOffice: 'Beşiktaş',
        email: 'info@abcteknoloji.com',
        phone: '+90 212 555 0123',
        address: 'Levent Mahallesi, İş Kuleleri No:15 Kat:10',
        city: 'İstanbul',
        country: 'Türkiye',
        contactPerson: 'Fatma Kaya',
        department: 'İnsan Kaynakları',
        isVip: true,
        isActive: true,
        notes: 'Çalışan seyahatleri için anlaşmalı firma. Aylık 20-30 bilet talebi var.',
        createdAt: new Date('2024-03-10'),
        updatedAt: new Date('2024-12-18')
      };
      
      setCustomer(mockCustomer);
    } catch (error) {
      console.error('Error fetching customer details:', error);
    } finally {
      setLoading(false);
    }
  };

  const fetchComments = async () => {
    const mockComments: Comment[] = [
      {
        id: '1',
        author: 'Sistem',
        content: 'Müşteri kaydı oluşturuldu',
        timestamp: new Date('2024-03-10T10:00:00'),
        type: 'system'
      },
      {
        id: '2',
        author: 'Agent Mehmet',
        content: 'İlk görüşme yapıldı, anlaşma imzalandı',
        timestamp: new Date('2024-03-10T14:30:00'),
        type: 'comment'
      },
      {
        id: '3',
        author: 'Fatma Kaya',
        content: 'Merhaba, bu ay 5 çalışanımız için bilet talep edeceğiz',
        timestamp: new Date('2024-12-15T09:00:00'),
        type: 'comment'
      },
      {
        id: '4',
        author: 'Agent Mehmet',
        content: 'VIP statüsü verildi - yüksek hacimli müşteri',
        timestamp: new Date('2024-12-18T11:00:00'),
        type: 'comment',
        isInternal: true
      }
    ];
    setComments(mockComments);
  };

  const fetchServices = async () => {
    const mockServices: CustomerService[] = [
      {
        id: '1',
        type: 'ticket',
        description: 'THY TK1234 - IST → ADB (5 yolcu)',
        amount: 4250,
        currency: 'TRY',
        status: 'completed',
        date: new Date('2024-12-15')
      },
      {
        id: '2',
        type: 'accommodation',
        description: 'Bodrum Bay Resort - 3 gece (2 oda)',
        amount: 6800,
        currency: 'TRY',
        status: 'confirmed',
        date: new Date('2024-12-10')
      },
      {
        id: '3',
        type: 'visa',
        description: 'ABD İş Vizesi (3 kişi)',
        amount: 1980,
        currency: 'USD',
        status: 'processing',
        date: new Date('2024-12-08')
      },
      {
        id: '4',
        type: 'transfer',
        description: 'Havalimanı Transfer - VIP Araç',
        amount: 450,
        currency: 'TRY',
        status: 'completed',
        date: new Date('2024-12-05')
      }
    ];
    setServices(mockServices);
  };

  const handleAddComment = (content: string, isInternal: boolean) => {
    const newComment: Comment = {
      id: Date.now().toString(),
      author: 'Agent Mehmet',
      content,
      timestamp: new Date(),
      type: 'comment',
      isInternal
    };
    setComments(prev => [...prev, newComment]);
  };

  const handleStatusChange = (newStatus: string) => {
    if (!customer) return;
    
    const isActive = newStatus === 'active';
    setCustomer(prev => prev ? { ...prev, isActive, updatedAt: new Date() } : null);
    
    const statusComment: Comment = {
      id: Date.now().toString(),
      author: 'Sistem',
      content: `Müşteri durumu "${isActive ? 'Aktif' : 'Pasif'}" olarak güncellendi`,
      timestamp: new Date(),
      type: 'status_change'
    };
    setComments(prev => [...prev, statusComment]);
  };

  const statusOptions = [
    { value: 'active', label: 'Aktif' },
    { value: 'inactive', label: 'Pasif' }
  ];

  // İstatistikler
  const stats = {
    totalServices: services.length,
    totalAmount: services.reduce((sum, service) => sum + service.amount, 0),
    completedServices: services.filter(s => s.status === 'completed').length,
    pendingServices: services.filter(s => s.status !== 'completed').length
  };

  if (loading || !customer) {
    return (
      <div className="min-h-screen bg-gray-50 flex items-center justify-center">
        <div className="animate-pulse space-y-4">
          <div className="h-8 bg-gray-200 rounded w-64"></div>
          <div className="h-64 bg-gray-200 rounded w-96"></div>
        </div>
      </div>
    );
  }

  return (
    <DetailLayout
      title={customer.type === 'individual' 
        ? `${customer.firstName} ${customer.lastName}` 
        : customer.companyName || 'Müşteri'
      }
      subtitle={customer.type === 'corporate' 
        ? `${customer.contactPerson} - ${customer.department}` 
        : customer.email
      }
      backUrl="/customers"
      recordId={customer.id}
      recordType="ticket"
      comments={comments}
      onAddComment={handleAddComment}
      onStatusChange={handleStatusChange}
      statusOptions={statusOptions}
      currentStatus={customer.isActive ? 'active' : 'inactive'}
    >
      <div className="space-y-6">
        {/* Customer Info Card */}
        <div className="bg-white rounded-lg shadow-sm border border-gray-200 p-6">
          <div className="flex items-center justify-between mb-6">
            <h3 className="text-lg font-semibold text-gray-900 flex items-center">
              {customer.type === 'corporate' ? (
                <Building2 className="h-5 w-5 mr-2 text-purple-600" />
              ) : (
                <User className="h-5 w-5 mr-2 text-blue-600" />
              )}
              Müşteri Bilgileri
            </h3>
            <div className="flex items-center space-x-2">
              {customer.isVip && (
                <span className="px-2 py-1 text-xs bg-yellow-100 text-yellow-800 rounded-full flex items-center">
                  <Crown className="h-3 w-3 mr-1" />
                  VIP
                </span>
              )}
              <span className={`px-2 py-1 text-xs rounded-full ${
                customer.isActive 
                  ? 'bg-green-100 text-green-800' 
                  : 'bg-red-100 text-red-800'
              }`}>
                {customer.isActive ? 'Aktif' : 'Pasif'}
              </span>
            </div>
          </div>

          <div className="grid grid-cols-1 md:grid-cols-2 gap-6">
            <div className="space-y-4">
              <div className="flex items-center space-x-3">
                <Mail className="h-4 w-4 text-gray-400" />
                <div>
                  <p className="text-sm text-gray-500">E-posta</p>
                  <p className="font-medium">{customer.email}</p>
                </div>
              </div>
              
              <div className="flex items-center space-x-3">
                <Phone className="h-4 w-4 text-gray-400" />
                <div>
                  <p className="text-sm text-gray-500">Telefon</p>
                  <p className="font-medium">{customer.phone}</p>
                </div>
              </div>
              
              <div className="flex items-center space-x-3">
                <MapPin className="h-4 w-4 text-gray-400" />
                <div>
                  <p className="text-sm text-gray-500">Konum</p>
                  <p className="font-medium">{customer.city}{customer.country && `, ${customer.country}`}</p>
                </div>
              </div>
            </div>

            <div className="space-y-4">
              {customer.type === 'corporate' && (
                <>
                  <div className="flex items-center space-x-3">
                    <FileText className="h-4 w-4 text-gray-400" />
                    <div>
                      <p className="text-sm text-gray-500">Vergi No</p>
                      <p className="font-medium">{customer.taxNumber}</p>
                    </div>
                  </div>
                  
                  <div className="flex items-center space-x-3">
                    <Building2 className="h-4 w-4 text-gray-400" />
                    <div>
                      <p className="text-sm text-gray-500">Vergi Dairesi</p>
                      <p className="font-medium">{customer.taxOffice}</p>
                    </div>
                  </div>
                </>
              )}
              
              <div className="flex items-center space-x-3">
                <Calendar className="h-4 w-4 text-gray-400" />
                <div>
                  <p className="text-sm text-gray-500">Kayıt Tarihi</p>
                  <p className="font-medium">{customer.createdAt.toLocaleDateString('tr-TR')}</p>
                </div>
              </div>
            </div>
          </div>

          {customer.address && (
            <div className="mt-6 pt-6 border-t border-gray-200">
              <p className="text-sm text-gray-500 mb-2">Adres</p>
              <p className="text-gray-900">{customer.address}</p>
            </div>
          )}

          {customer.notes && (
            <div className="mt-6 pt-6 border-t border-gray-200">
              <p className="text-sm text-gray-500 mb-2">Notlar</p>
              <p className="text-gray-700">{customer.notes}</p>
            </div>
          )}
        </div>

        {/* Stats Cards */}
        <div className="grid grid-cols-1 md:grid-cols-4 gap-4">
          <div className="bg-white p-4 rounded-lg border border-gray-200">
            <div className="flex items-center justify-between">
              <div>
                <p className="text-sm text-gray-600">Toplam Hizmet</p>
                <p className="text-2xl font-bold text-gray-900">{stats.totalServices}</p>
              </div>
              <FileText className="h-8 w-8 text-gray-400" />
            </div>
          </div>
          
          <div className="bg-white p-4 rounded-lg border border-gray-200">
            <div className="flex items-center justify-between">
              <div>
                <p className="text-sm text-gray-600">Toplam Tutar</p>
                <p className="text-2xl font-bold text-green-600">
                  {stats.totalAmount.toLocaleString()} TRY
                </p>
              </div>
              <CreditCard className="h-8 w-8 text-green-400" />
            </div>
          </div>
          
          <div className="bg-white p-4 rounded-lg border border-gray-200">
            <div className="flex items-center justify-between">
              <div>
                <p className="text-sm text-gray-600">Tamamlanan</p>
                <p className="text-2xl font-bold text-blue-600">{stats.completedServices}</p>
              </div>
              <TrendingUp className="h-8 w-8 text-blue-400" />
            </div>
          </div>
          
          <div className="bg-white p-4 rounded-lg border border-gray-200">
            <div className="flex items-center justify-between">
              <div>
                <p className="text-sm text-gray-600">Devam Eden</p>
                <p className="text-2xl font-bold text-orange-600">{stats.pendingServices}</p>
              </div>
              <Clock className="h-8 w-8 text-orange-400" />
            </div>
          </div>
        </div>

        {/* Services List */}
        <div className="bg-white rounded-lg shadow-sm border border-gray-200 p-6">
          <div className="flex items-center justify-between mb-4">
            <h3 className="text-lg font-semibold text-gray-900">Son Hizmetler</h3>
            <Link
              href={`/customers/${customer.id}/services`}
              className="text-blue-600 hover:text-blue-800 text-sm"
            >
              Tümünü Gör
            </Link>
          </div>
          
          <div className="space-y-4">
            {services.slice(0, 5).map((service) => (
              <div key={service.id} className="flex items-center justify-between p-4 bg-gray-50 rounded-lg">
                <div className="flex items-center space-x-3">
                  {renderServiceType(service.type)}
                  <div>
                    <p className="font-medium text-gray-900">{service.description}</p>
                    <p className="text-sm text-gray-500">{service.date.toLocaleDateString('tr-TR')}</p>
                  </div>
                </div>
                <div className="text-right">
                  <p className="font-medium text-gray-900">
                    {renderCurrency(service.amount, service.currency)}
                  </p>
                  {renderStatusBadge(service.status, {
                    completed: { color: 'bg-green-100 text-green-800', label: 'Tamamlandı' },
                    confirmed: { color: 'bg-blue-100 text-blue-800', label: 'Onaylandı' },
                    processing: { color: 'bg-yellow-100 text-yellow-800', label: 'İşlemde' },
                    pending: { color: 'bg-gray-100 text-gray-800', label: 'Bekliyor' }
                  })}
                </div>
              </div>
            ))}
          </div>
        </div>

        {/* Quick Actions */}
        <div className="bg-white rounded-lg shadow-sm border border-gray-200 p-6">
          <h3 className="text-lg font-semibold text-gray-900 mb-4">Hızlı İşlemler</h3>
          <div className="grid grid-cols-2 md:grid-cols-4 gap-4">
            <Link
              href={`/tickets/new?customer=${customer.id}`}
              className="flex flex-col items-center p-4 border border-gray-200 rounded-lg hover:bg-gray-50 transition-colors"
            >
              <Plane className="h-8 w-8 text-blue-600 mb-2" />
              <span className="text-sm font-medium">Bilet Talebi</span>
            </Link>
            
            <Link
              href={`/accommodation/requests/new?customer=${customer.id}`}
              className="flex flex-col items-center p-4 border border-gray-200 rounded-lg hover:bg-gray-50 transition-colors"
            >
              <Hotel className="h-8 w-8 text-purple-600 mb-2" />
              <span className="text-sm font-medium">Konaklama</span>
            </Link>
            
            <Link
              href={`/visa/requests/new?customer=${customer.id}`}
              className="flex flex-col items-center p-4 border border-gray-200 rounded-lg hover:bg-gray-50 transition-colors"
            >
              <Globe className="h-8 w-8 text-indigo-600 mb-2" />
              <span className="text-sm font-medium">Vize Başvuru</span>
            </Link>
            
            <Link
              href={`/transfer/requests/new?customer=${customer.id}`}
              className="flex flex-col items-center p-4 border border-gray-200 rounded-lg hover:bg-gray-50 transition-colors"
            >
              <Car className="h-8 w-8 text-green-600 mb-2" />
              <span className="text-sm font-medium">Transfer</span>
            </Link>
          </div>
        </div>

        {/* Corporate Users (if corporate customer) */}
        {customer.type === 'corporate' && (
          <div className="bg-white rounded-lg shadow-sm border border-gray-200 p-6">
            <div className="flex items-center justify-between mb-4">
              <h3 className="text-lg font-semibold text-gray-900 flex items-center">
                <Users className="h-5 w-5 mr-2" />
                Kurumsal Kullanıcılar
              </h3>
              <Link
                href={`/customers/${customer.id}/users`}
                className="flex items-center px-3 py-2 text-sm text-blue-600 border border-blue-300 rounded-lg hover:bg-blue-50"
              >
                <Settings className="h-4 w-4 mr-1" />
                Yönet
              </Link>
            </div>
            <p className="text-gray-600">
              Bu kurumsal müşterinin kullanıcılarını yönetmek için yukarıdaki butonu kullanın.
            </p>
          </div>
        )}

        {/* Actions */}
        <div className="bg-white rounded-lg shadow-sm border border-gray-200 p-6">
          <h3 className="text-lg font-semibold text-gray-900 mb-4">İşlemler</h3>
          <div className="flex flex-wrap gap-3">
            <Link
              href={`/customers/${customer.id}/edit`}
              className="flex items-center px-4 py-2 bg-blue-600 text-white rounded-lg hover:bg-blue-700"
            >
              <Edit className="h-4 w-4 mr-2" />
              Düzenle
            </Link>
            
            <button className="flex items-center px-4 py-2 bg-gray-600 text-white rounded-lg hover:bg-gray-700">
              <FileText className="h-4 w-4 mr-2" />
              Rapor Al
            </button>
            
            <button className="flex items-center px-4 py-2 bg-green-600 text-white rounded-lg hover:bg-green-700">
              <Phone className="h-4 w-4 mr-2" />
              Ara
            </button>
            
            <button className="flex items-center px-4 py-2 bg-purple-600 text-white rounded-lg hover:bg-purple-700">
              <Mail className="h-4 w-4 mr-2" />
              E-posta Gönder
            </button>
          </div>
        </div>
      </div>
    </DetailLayout>
  );
}